/*
 * Decompiled with CFR 0.152.
 */
package nom.tam.fits;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.Locale;
import java.util.StringTokenizer;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.regex.Pattern;
import nom.tam.fits.FitsFactory;
import nom.tam.fits.HeaderCard;
import nom.tam.fits.UnclosedQuoteException;
import nom.tam.fits.header.NonStandard;
import nom.tam.fits.header.Standard;
import nom.tam.util.ComplexValue;

class HeaderCardParser {
    private static final Logger LOG = Logger.getLogger(HeaderCardParser.class.getName());
    private static final Pattern DECIMAL_REGEX;
    private static final Pattern COMPLEX_REGEX;
    private static final Pattern INT_REGEX;
    private String line;
    private String key = null;
    private String value = null;
    private String comment = null;
    private Class<?> type = null;
    private int parsePos = 0;

    HeaderCardParser(String line) throws UnclosedQuoteException, IllegalArgumentException {
        this.line = line;
        this.parseKey();
        this.parseValue();
        this.parseComment();
    }

    String getKey() {
        return this.key;
    }

    String getValue() {
        return this.value;
    }

    String getUntrimmedComment() {
        return this.comment;
    }

    String getTrimmedComment() {
        return this.comment == null ? null : this.comment.trim();
    }

    boolean isString() {
        if (this.type == null) {
            return false;
        }
        return String.class.isAssignableFrom(this.type);
    }

    Class<?> getInferredType() {
        return this.type;
    }

    private void parseKey() {
        String stem;
        int iEq = this.line.indexOf(61);
        int endStem = iEq >= 0 && iEq <= 8 ? iEq : 8;
        endStem = Math.min(this.line.length(), endStem);
        String rawStem = this.line.substring(0, endStem).trim();
        if (endStem > 0 && !rawStem.isEmpty() && Character.isSpaceChar(this.line.charAt(0))) {
            LOG.warning("[" + HeaderCardParser.sanitize(rawStem) + "] Non-standard starting with a space (trimming).");
        }
        if (!(stem = rawStem.toUpperCase()).equals(rawStem)) {
            LOG.warning("[" + HeaderCardParser.sanitize(rawStem) + "] Non-standard lower-case letter(s) in base keyword.");
        }
        this.key = stem;
        this.parsePos = endStem;
        if (!FitsFactory.getUseHierarch() || iEq < 0 || !stem.equals(NonStandard.HIERARCH.key())) {
            return;
        }
        StringTokenizer tokens = new StringTokenizer(this.line.substring(stem.length(), iEq), " \t\r\n.");
        StringBuilder builder = new StringBuilder(stem);
        while (tokens.hasMoreTokens()) {
            String token = tokens.nextToken();
            this.parsePos = this.line.indexOf(token, this.parsePos) + token.length();
            builder.append('.');
            builder.append(token);
        }
        this.key = builder.toString();
        if (NonStandard.HIERARCH.key().equals(this.key)) {
            LOG.warning("HIERARCH base keyword without HIERARCH-style long key after it.");
            return;
        }
        if (!FitsFactory.getHierarchFormater().isCaseSensitive()) {
            this.key = this.key.toUpperCase(Locale.US);
        }
        try {
            HeaderCard.validateKey(this.key);
        }
        catch (IllegalArgumentException e) {
            LOG.warning(e.getMessage());
        }
    }

    private boolean skipSpaces() {
        while (this.parsePos < this.line.length()) {
            if (!Character.isSpaceChar(this.line.charAt(this.parsePos))) {
                return true;
            }
            ++this.parsePos;
        }
        return false;
    }

    private void parseComment() {
        if (!this.skipSpaces()) {
            return;
        }
        if (this.value != null) {
            if (this.line.charAt(this.parsePos) == '/') {
                ++this.parsePos;
            } else {
                LOG.warning("[" + HeaderCardParser.sanitize(this.getKey()) + "] Junk after value (included in the comment).");
            }
        }
        this.comment = this.line.substring(this.parsePos);
        this.parsePos = this.line.length();
        try {
            HeaderCard.validateChars(this.comment);
        }
        catch (IllegalArgumentException e) {
            LOG.warning("[" + HeaderCardParser.sanitize(this.getKey()) + "]: " + e.getMessage());
        }
    }

    private void parseValue() throws UnclosedQuoteException {
        if (this.key.isEmpty() || !this.skipSpaces()) {
            return;
        }
        if (Standard.CONTINUE.key().equals(this.key)) {
            this.parseValueBody();
        } else if (this.line.charAt(this.parsePos) == '=') {
            if (this.parsePos < 8) {
                LOG.warning("[" + HeaderCardParser.sanitize(this.key) + "] assigmment before byte " + 9 + " for key '" + HeaderCardParser.sanitize(this.key) + "'.");
            }
            if (this.parsePos + 1 >= this.line.length()) {
                LOG.warning("[" + HeaderCardParser.sanitize(this.key) + "] Record ends with '='.");
            } else if (this.line.charAt(this.parsePos + 1) != ' ') {
                LOG.warning("[" + HeaderCardParser.sanitize(this.key) + "] Missing required standard space after '='.");
            }
            if (this.parsePos > 8 && !this.key.startsWith(NonStandard.HIERARCH.key() + ".")) {
                LOG.warning("[" + HeaderCardParser.sanitize(this.key) + "] Possibly misplaced '=' (after byte 9).");
                return;
            }
            ++this.parsePos;
            this.parseValueBody();
        }
        try {
            HeaderCard.validateChars(this.value);
        }
        catch (IllegalArgumentException e) {
            LOG.warning("[" + HeaderCardParser.sanitize(this.getKey()) + "] " + e.getMessage());
        }
    }

    private void parseValueBody() throws UnclosedQuoteException {
        if (!this.skipSpaces()) {
            return;
        }
        if (this.isNextQuote()) {
            this.parseStringValue();
        } else {
            int end = this.line.indexOf(47, this.parsePos);
            if (end < 0) {
                end = this.line.length();
            }
            this.value = this.line.substring(this.parsePos, end).trim();
            this.parsePos = end;
            this.type = HeaderCardParser.getInferredValueType(this.key, this.value);
        }
    }

    private boolean isNextQuote() {
        if (this.parsePos >= this.line.length()) {
            return false;
        }
        return this.line.charAt(this.parsePos) == '\'';
    }

    private static String getNoTrailingSpaceString(StringBuilder buf) {
        int to = buf.length();
        while (--to >= 0 && Character.isSpaceChar(buf.charAt(to))) {
        }
        return to < 0 ? "" : buf.substring(0, to + 1);
    }

    private void parseStringValue() throws UnclosedQuoteException {
        this.type = String.class;
        StringBuilder buf = new StringBuilder(70);
        ++this.parsePos;
        while (this.parsePos < this.line.length()) {
            if (this.isNextQuote()) {
                ++this.parsePos;
                if (!this.isNextQuote()) {
                    this.value = HeaderCardParser.getNoTrailingSpaceString(buf);
                    return;
                }
            }
            buf.append(this.line.charAt(this.parsePos));
            ++this.parsePos;
        }
        if (!FitsFactory.isAllowHeaderRepairs()) {
            throw new UnclosedQuoteException(this.line);
        }
        LOG.warning("[" + HeaderCardParser.sanitize(this.key) + "] Ignored missing end quote (value parsed to end of record).");
        this.value = HeaderCardParser.getNoTrailingSpaceString(buf);
    }

    private static Class<?> getInferredValueType(String key, String value) {
        if (value.isEmpty()) {
            LOG.warning("[" + HeaderCardParser.sanitize(key) + "] Null non-string value (defaulted to Boolean.class).");
            return Boolean.class;
        }
        String trimmedValue = value.trim().toUpperCase();
        if ("T".equals(trimmedValue) || "F".equals(trimmedValue)) {
            return Boolean.class;
        }
        if (INT_REGEX.matcher(trimmedValue).matches()) {
            return HeaderCardParser.getIntegerType(trimmedValue);
        }
        if (DECIMAL_REGEX.matcher(trimmedValue).matches()) {
            return HeaderCardParser.getDecimalType(trimmedValue);
        }
        if (COMPLEX_REGEX.matcher(trimmedValue).matches()) {
            return ComplexValue.class;
        }
        LOG.warning("[" + HeaderCardParser.sanitize(key) + "] Unrecognised non-string value type '" + HeaderCardParser.sanitize(trimmedValue) + "'.");
        return null;
    }

    private static Class<? extends Number> getDecimalType(String value) {
        BigDecimal big;
        boolean hasD;
        boolean bl = hasD = (value = value.toUpperCase(Locale.US)).indexOf(68) >= 0;
        if (hasD) {
            value = value.replace('D', 'E');
        }
        if ((big = new BigDecimal(value)).stripTrailingZeros().equals(BigDecimal.ZERO)) {
            int decimals = big.scale();
            if (decimals <= 7) {
                return hasD ? Double.class : Float.class;
            }
            if (decimals <= 16) {
                return Double.class;
            }
            return BigDecimal.class;
        }
        int decimals = big.precision() - 1;
        float f = big.floatValue();
        if (decimals <= 7 && f != 0.0f && Float.isFinite(f)) {
            return hasD ? Double.class : Float.class;
        }
        double d = big.doubleValue();
        if (decimals <= 16 && d != 0.0 && Double.isFinite(d)) {
            return Double.class;
        }
        return BigDecimal.class;
    }

    private static Class<? extends Number> getIntegerType(String value) {
        int bits = new BigInteger(value).bitLength();
        if (bits < 32) {
            return Integer.class;
        }
        if (bits < 64) {
            return Long.class;
        }
        return BigInteger.class;
    }

    private static String sanitize(String text) {
        return HeaderCard.sanitize(text);
    }

    static Logger getLogger() {
        return LOG;
    }

    static {
        LOG.setLevel(Level.SEVERE);
        DECIMAL_REGEX = Pattern.compile("[+-]?\\d+(\\.\\d*)?([dDeE][+-]?\\d+)?");
        COMPLEX_REGEX = Pattern.compile("\\(\\s*" + DECIMAL_REGEX + "\\s*,\\s*" + DECIMAL_REGEX + "\\s*\\)");
        INT_REGEX = Pattern.compile("[+-]?\\d+");
    }
}

